#include <concepts>
#include <optional>
#include <map>
#include <string_view>
#include <stdexcept>
#include <memory>
#include <vector>
#include <variant>

namespace rpn {

/* we will use `stack` to represent the stack of a rpn calculation */
using stack = std::vector< long >;

/* and we will use `arguments` to represent arguments to the rpn operations */
using arguments = std::vector< long >;

/* finally, we will use `program` for the program consisting of numbers and operations */
using num_or_op = std::variant< long, std::string_view >;
using program = std::vector< num_or_op >;

struct base_op {
    /* invoke the operation on exactly `arity()` arguments */
    virtual long invoke( const arguments & ) const = 0;

    /* returns arity of the operation, this must be constant for a given
     * descendant type */
    virtual int arity() const = 0;

    virtual ~base_op() = default;
};

namespace ops {
    struct binary : base_op {
        /* use override to ensure we are not shadowing accidentally */
        int arity() const override { return 2; }
    };

    struct add : binary {
        long invoke( const arguments &args ) const override {
            return args[0] + args[1];
        }
    };

    struct sub : binary {
        long invoke( const arguments &args ) const override {
            return args[0] - args[1];
        }
    };

    struct mul : binary {
        long invoke( const arguments &args ) const override {
            return args[0] * args[1];
        }
    };

    struct div : binary {
        long invoke( const arguments &args ) const override {
            return args[0] / args[1];
        }
    };

    struct unary : base_op {
        int arity() const override { return 1; }
    };

    struct neg : unary {
        long invoke( const arguments &args ) const override {
            return - args[0];
        }
    };
}

struct exception : std::runtime_error {
    /* inherit constructors */
    using std::runtime_error::runtime_error;
};

struct calculator {

    calculator() = default;
    calculator( const calculator & ) = delete;

    // TODO
    /* Add an operation to the calculator. If an operation with given name
     * already exists, it is unmodified and `rpn::exception` is raised.
     * Returns `*this` so that the registrations can be chained */
    calculator &register_op( std::string_view name, std::unique_ptr< base_op > op );

    /* The calculator keeps (on top of its operations registered with
     * `register_op` also a stack of values. The content of the stack can be
     * deleted with this function. */
    void reset_stack() { _stack.clear(); }

    // TODO
    /* Get the top-most value on the stack, if there is any */
    std::optional< long > get_top() const;

    /* Get a copy of the current stack */
    stack get_stack() const { return _stack; }

    /* The `invoke_op` overloads can be used to run a single operation on the
     * RPN machine. This operation can be either "push integer" (given a `long`
     * value), or "invoke operation by name" given a `std::string_view` with a
     * name of the operation. The function returns a `bool` that indicates if
     * the operation was run successfully. */

    // TODO
    /* Invoke and operation "push integer". Always returns `true`, this
     * operation cannot fail. */
    bool invoke_op( long val );

    // TODO
    /* Invoke an operation by name. It succeeds if the name was registered and
     * if there were enough arguments on the stack for the operation. Otherwise
     * it fails. If the invocation fails, the stack is unmodified. */
    bool invoke_op( std::string_view opstr );

    // TODO
    /* Invoke the build rpn calculation on a given program. This modifies the
     * internal stack of the calculator. Returns `std::nullopt` if the program
     * is malformed (e.g. uses unsupported operation or goes into a state where
     * there is not enough arguments). Otherwise returns the value on top of
     * the stack at the end of the computation. If an error occurs, the
     * processing of program stops immediately and no further modifications of
     * the stack are performed. */
    std::optional< long > invoke( const program &prog );

  private:
    std::map< std::string_view, std::unique_ptr< base_op > > _ops;
    stack _stack;
};

} // end namespace rpn
