#ifdef EXAMPLE_SOLUTION
#include "rpn_solution.h"
#else
#include "rpn.h"
#endif
#define CATCH_CONFIG_MAIN
#include <catch2/catch.hpp>

TEST_CASE( "create calculator" ) {
    rpn::calculator calc;
    REQUIRE( calc.get_top() == std::nullopt );
}

TEST_CASE( "operator addition" ) {
    rpn::calculator calc;
    auto &r = calc.register_op( "+", std::make_unique< rpn::ops::add >() );
    REQUIRE( &r == &calc );
    REQUIRE( calc.get_top() == std::nullopt );
}

TEST_CASE( "add numbers to stack" ) {
    rpn::calculator calc;
    calc.invoke_op( 42 );
    REQUIRE( calc.get_top() == 42 );
    calc.invoke_op( 16 );
    REQUIRE( calc.get_top() == 16 );
    REQUIRE( calc.get_stack() == rpn::stack{ 42, 16 } );
}

TEST_CASE( "add works" ) {
    rpn::ops::add a;
    REQUIRE( a.invoke( { 40, 2 } ) == 42 );
}

TEST_CASE( "invoke single operation" ) {
    rpn::calculator calc;
    calc.register_op( "+", std::make_unique< rpn::ops::add >() );
    calc.invoke_op( 21 );
    calc.invoke_op( 21 );
    calc.invoke_op( "+" );
    REQUIRE( calc.get_top() == 42 );
    REQUIRE( calc.get_stack() == rpn::stack{ 42 } );
}

TEST_CASE( "invoke a simple program with numbers only" ) {
    rpn::calculator calc;
    calc.invoke( { 1, 2, 3, 4, 5 } );
    REQUIRE( calc.get_top() == 5 );
    REQUIRE( calc.get_stack() == rpn::stack{ 1, 2, 3, 4, 5 } );
}

TEST_CASE( "invoke a simple program with one op" ) {
    rpn::calculator calc;
    calc.register_op( "+", std::make_unique< rpn::ops::add >() );
    auto r = calc.invoke( { 20, 22, "+" } );
    REQUIRE( r.has_value() );
    REQUIRE( *r == 42 );
    REQUIRE( calc.get_top() == 42 );
    REQUIRE( calc.get_stack() == rpn::stack{ 42 } );
}

TEST_CASE( "invoke a program with two additions" ) {
    rpn::calculator calc;
    calc.register_op( "+", std::make_unique< rpn::ops::add >() );
    std::optional< long > r;
    SECTION( "add, push, add" ) {
        r = calc.invoke( { 20, 12, "+", 10, "+" } );
    }
    SECTION( "push, add, add" ) {
        r = calc.invoke( { 20, 12, 10, "+", "+" } );
    }
    REQUIRE( r.has_value() );
    REQUIRE( *r == 42 );
    REQUIRE( calc.get_top() == 42 );
    REQUIRE( calc.get_stack() == rpn::stack{ 42 } );
}

TEST_CASE( "a more complex programs" ) {
    rpn::calculator calc;
    calc.register_op( "+", std::make_unique< rpn::ops::add >() )
        .register_op( "-", std::make_unique< rpn::ops::sub >() )
        .register_op( "*", std::make_unique< rpn::ops::mul >() )
        .register_op( "/", std::make_unique< rpn::ops::div >() )
        .register_op( "neg", std::make_unique< rpn::ops::neg >() );
    SECTION( "multiops 1" ) {
        auto r = calc.invoke( { 2, 10, 4, "*", "+" } );
        REQUIRE( r.has_value() );
        REQUIRE( *r == 42 );
    }
    SECTION( "multiops 2 – non-comutative" ) {
        auto r = calc.invoke( { 20, 20, "*", 10, "/", 2, "+" } );
        REQUIRE( r.has_value() );
        REQUIRE( *r == 42 );
    }
    SECTION( "multiple invokes" ) {
        auto r = calc.invoke( { 2, 2, 2, 2, 2, "*", "*" } );
        REQUIRE( r.has_value() );
        REQUIRE( *r == 8 );
        r = calc.invoke( { "*", "*" } );
        REQUIRE( r.has_value() );
        REQUIRE( *r == 32 );
    }
    SECTION( "not enough args 1" ) {
        auto r = calc.invoke( { 4, "*" } );
        REQUIRE_FALSE( r.has_value() );
    }
    SECTION( "not enough args 2" ) {
        auto r = calc.invoke( { "neg" } );
        REQUIRE_FALSE( r.has_value() );
    }
    SECTION( "not enough args 3" ) {
        auto r = calc.invoke( { 2, 2, 2, "+", "+", "+" } );
        REQUIRE_FALSE( r.has_value() );
    }
    SECTION( "unknown op" ) {
        auto r = calc.invoke( { "undefined" } );
        REQUIRE_FALSE( r.has_value() );
    }
}

TEST_CASE( "register twice" ) {
    rpn::calculator calc;
    calc.register_op( "+", std::make_unique< rpn::ops::add >() );
    REQUIRE_THROWS( calc.register_op( "+", std::make_unique< rpn::ops::sub >() ) );
}
