#pragma once

#include <concepts>
#include <vector>

/* ArraySet is an implementation of set that uses sorted array as its backend.
 * The advantage of ArraySet is that is has very fast iteration and lookup.
 * The disadvantage is that insertion is liner (if the element is not already
 * present).
 * This makes ArraySet especially useful for smaller data sizes (since it can
 * fit into cache and the CPU does not need to chase pointers).
 * The interface of ArraySet is a simplification of the interace of std::set,
 * with addition of operator[]. The iterators need not have the same guarantees
 * as for std::set, i.e. they can be invalidated by insertion.
 *
 * You should implement the parts marked with TODO. In some cases we also
 * specify complexity for given member function (or a group of them).
 * In these cases your solution should reflect this requirement.
 *
 * TODO:
 * ArraySet is parametrized by a single template argument T that must be
 * comparable, move constructible, and movable. This must be ensured using
 * concepts.
 */
template /* ... */
struct ArraySet {

    using value_type = T;
    using key_type = T;
    /* we use vector to store the data (see below), so just reuse some nested
     * types from vector */
    using size_type = typename std::vector< T >::size_type;
    using iterator = typename std::vector< T >::iterator;
    using const_iterator = typename std::vector< T >::const_iterator;

    // O(1)
    ArraySet() = default;

    // O(n log n)
    /* TODO:
     * Constructs an ArraySet from an (unsorted) range given by two iterators.
     * The template must use concepts to ensure that from and to are iterators.
     * Bonus: Ensure the iterarors point to types from which T can be
     * constructed.
     */
    template /* ... */
    ArraySet( /* ... */ from, /* ... */ to );

    // O(n log n)
    ArraySet( std::initializer_list< T > init ) :
        ArraySet( init.begin(), init.end() )
    { }

    // O(n)
    /* Automatically derive all equality and comparison operators. */
    auto operator<=>( const ArraySet & ) const = default;

    // O(1)
    /* Iterators just point to the underlying vector. */
    auto begin() { return _data.begin(); }
    auto begin() const { return _data.begin(); }
    auto cbegin() const { return _data.begin(); }

    auto end() { return _data.end(); }
    auto end() const { return _data.end(); }
    auto cend() const { return _data.end(); }

    // O(1)
    bool empty() const { return _data.empty(); }

    // O(1)
    size_t size() const { return _data.size(); }

    // O(n)
    void clear() { _data.clear(); }

    // O(log n) if the element is already present,
    // O(n) otherwise (due complexity of insert on vector)
    /* TODO:
     * Implement insertion with the aforementioned complexities and semantics
     * mathiching that of std::set::insert with the same prototype.
     */
    std::pair< iterator, bool > insert( const value_type &value );

    // O(log n)
    bool contains( const value_type &value ) const {
        return find( value ) != end();
    }

    // O(log n)
    size_type count( const value_type &value ) const {
        return contains( value );
    }

    // O(log n)
    iterator find( const value_type &value ) {
        // to avoid duplication of code in the const and non-const version
        // of find, we defer to a helper that is templated on the first
        // argument – it will be either ArraySet & or const ArraySet &
        return _find( *this, value );
    }

    // O(log n)
    const_iterator find( const value_type &value ) const {
        // see above
        return _find( *this, value );
    }

    // O(n)
    iterator erase( iterator pos ) { return _data.erase( pos ); }

    // O(log n) if the element is not present O(n) otherwise
    /* TODO:
     * Implement erase-by-value with the aforementioned complexity.
     */
    size_type erase( const value_type &value );

    value_type &operator[]( size_type idx ) { return _data[ idx ]; }
    const value_type &operator[]( size_type idx ) const { return _data[ idx ]; }

  private:
    // O(log n)
    /* TODO:
     * This is the actual implementation of find. Implement it with the
     * aforementioned complexity.
     * To make it work with both const and non-const this we define it it a
     * static function which takes this as a parametr -- self can be either
     * const ArraySet & or ArraySet &.
     * Please note that the return type is derived -- it is either iterator or
     * const_iterator.
     */
    template< typename Self >
    static auto _find( Self &self, const value_type &value );

    std::vector< T > _data;
};

// vim: expandtab
