#ifdef EXAMPLE_SOLUTION
#include "arrayset_solution.h"
#else
#include "arrayset.h"
#endif
#define CATCH_CONFIG_MAIN
#include <catch2/catch.hpp>
#include <compare>

struct Ordered {
    std::strong_ordering operator<=>(const Ordered &) const = default;
};

struct Unordered {
};

template struct ArraySet< int >;
template struct ArraySet< Ordered >;
// template struct ArraySet< Unordered >; // Concept-based compilation error as required

TEST_CASE( "ctor empty" ) {
    ArraySet< int > s0;
    ArraySet< Ordered > s1;
    REQUIRE( s0.empty() );
    REQUIRE( s1.empty() );
    REQUIRE( s0.size() == 0 );
    REQUIRE( s1.size() == 0 );
}

static const std::initializer_list< int > range_0 = { 10, 3, 2, 15, 42 };

void check_ctor_range_0( auto &s0 ) {
    REQUIRE( *s0.begin() == 2 );
    REQUIRE( *std::next( s0.begin(), 1 ) == 3 );
    REQUIRE( *std::next( s0.begin(), 2 ) == 10 );
    REQUIRE( *std::next( s0.begin(), 3 ) == 15 );
    REQUIRE( *std::next( s0.begin(), 4 ) == 42 );
}

TEST_CASE( "ctor range" ) {
    std::vector< int > v0 = range_0;
    ArraySet< int > s0( v0.begin(), v0.end() );
    check_ctor_range_0( s0 );
}

TEST_CASE( "ctor ilist" ) {
    ArraySet< int > s0{ range_0 };
    check_ctor_range_0( s0 );
}

TEST_CASE( "copy & move & assign" ) {
    ArraySet< int > s0 = range_0;

    ArraySet< int > s1( s0 );
    check_ctor_range_0( s1 );

    ArraySet< int > s2 = { 1 };
    s2 = s1;
    check_ctor_range_0( s2 );

    ArraySet< int > s3( std::move( s2 ) );
    check_ctor_range_0( s3 );

    ArraySet< int > s4 = { 1 };
    s4 = std::move( s1 );
    check_ctor_range_0( s4 );
}

void check_sorted( auto &set ) {
    for ( auto it0 = set.begin(), it1 = std::next( set.begin() ), e = set.end();
          it0 != e && it1 != e; ++it0, ++it1 )
        REQUIRE( *it0 < *it1 );
}

TEST_CASE( "insert" ) {
    ArraySet< int > s = { 5, 2, 3 };
    check_sorted( s );
    auto [ it0, ins0 ] = s.insert( 42 );
    REQUIRE( ins0 );
    REQUIRE( *it0 == 42 );
    REQUIRE( std::next( it0 ) == s.end() );
    check_sorted( s );

    auto [ it1, ins1 ] = s.insert( 42 );
    REQUIRE_FALSE( ins1 );
    REQUIRE( *it1 == 42 );
    REQUIRE( it1 == it0 );
    check_sorted( s );

    auto [ it2, ins2 ] = s.insert( 0 );
    REQUIRE( ins2 );
    REQUIRE( *it2 == 0 );
    REQUIRE( it2 == s.begin() );
    check_sorted( s );

    auto [ it3, ins3 ] = s.insert( 3 );
    REQUIRE_FALSE( ins3 );
    REQUIRE( *it3 == 3 );
    check_sorted( s );

    auto [ it4, ins4 ] = s.insert( 4 );
    REQUIRE( ins4 );
    REQUIRE( *it4 == 4 );
    check_sorted( s );
}

TEST_CASE( "search" ) {
    ArraySet< int > s = { 10, 2, 42, 5 };

    SECTION( "find" ) {
        REQUIRE( s.find( 10 ) != s.end() );
        REQUIRE( *s.find( 10 ) == 10 );
        REQUIRE( s.find( 15 ) == s.end() );
    }
    SECTION( "count" ) {
        REQUIRE( s.count( 10 ) == 1 );
        REQUIRE( s.count( 15 ) == 0 );
    }
    SECTION( "contains" ) {
        REQUIRE( s.contains( 10 ) );
        REQUIRE_FALSE( s.contains( 15 ) );
    }
    check_sorted( s );
}

TEST_CASE( "erase" ) {
    ArraySet< int > s = { 10, 2, 42, 5 };
    SECTION( "by iterator" ) {
        auto it = s.erase( std::next( s.begin() ) );
        REQUIRE( *it == 10 );
    }
    SECTION( "by value" ) {
        REQUIRE( s.erase( 5 ) == 1 );
        REQUIRE( s.erase( 5 ) == 0 );
    }
    REQUIRE_FALSE( s.contains( 5 ) );
    REQUIRE( s.contains( 2 ) );
    REQUIRE( s.contains( 10 ) );
    REQUIRE( s.contains( 42 ) );
    REQUIRE( s.size() == 3 );
    check_sorted( s );

}
